(ns
    #^{:author "renarl",
       :doc "export to tda"}
  lv.lumii.tda.export
  (:use lv.lumii.tda
        lv.lumii.tda.utilities
        lv.lumii.tda.export.pipeline
        lv.lumii.tda.export.utilities
        lv.lumii.tda.export.renderer
				lv.lumii.tda.export.custom-render-spec
        lv.lumii.tda.serialize
        lv.lumii.tda.send
        lv.lumii.tda.export.progress-monitor
        clojure.set
        clj-stacktrace.repl)
  (:import
	 (java.io File BufferedWriter FileOutputStream OutputStreamWriter)
	 [uk.ac.manchester.cs.owlapi.modularity SyntacticLocalityModuleExtractor ModuleType]
   [org.semanticweb.owlapi.io FileDocumentSource]
   [org.semanticweb.owlapi.model MissingImportHandlingStrategy]
   [org.semanticweb.owlapi.model OWLOntologyLoaderConfiguration]
   [org.semanticweb.owlapi.model IRI OWLDeclarationAxiom
    OWLDataProperty
    OWLDisjointClassesAxiom OWLEquivalentClassesAxiom
    OWLDifferentIndividualsAxiom OWLSameIndividualAxiom
    OWLSubClassOfAxiom
    OWLClassAssertionAxiom OWLObjectPropertyAssertionAxiom OWLNegativeObjectPropertyAssertionAxiom OWLDataPropertyAssertionAxiom OWLNegativeDataPropertyAssertionAxiom
    OWLClass OWLNamedIndividual
    OWLObjectIntersectionOf
    OWLHasKeyAxiom
    OWLObjectUnionOf
    OWLSubPropertyChainOfAxiom
    OWLObjectSomeValuesFrom OWLObjectAllValuesFrom OWLObjectMinCardinality OWLObjectMaxCardinality OWLObjectExactCardinality
    OWLDataSomeValuesFrom OWLDataMinCardinality OWLDataMaxCardinality OWLDataExactCardinality]))

(defn- create-group-exporter [{:keys [name export-group-name axiom-type getter-fn group-keyword
                                      group-box-type-name entity-getter-fn box-label connection-line-type-name entity-renderer-fn
                                      group-line-type-name line-label
                                      entity-type-name expression-renderer-fn]}]
  [{:name (str "merge " name " links")
    :group export-group-name
    :before (groups-for-axiom-type axiom-type getter-fn group-keyword)
    :chunker (clique-getter-for-group-name group-keyword)
    :exporter (group-joiner-by-clique-for-group-name group-keyword)}
   {:name (str "mutualy " name " larger then 2 as boxes")
    :group export-group-name
    :chunker (fn [acc _] (set (filter (fn [group] (> (count group) 2)) (group-keyword acc))))
    :exporter (group-box-exporter entity-getter-fn group-box-type-name box-label connection-line-type-name group-keyword entity-renderer-fn)}
   {:name (str "pairs of " name " as lines")
    :group export-group-name
    :chunker (fn [acc _] (group-keyword acc))
    :exporter (group-line-exporter entity-getter-fn group-line-type-name line-label group-keyword)}
   {:name (str "other " name " as compartments")
    :group export-group-name
    :chunker (fn [acc _] (group-keyword acc))
    :exporter (group-compartment-exporter entity-getter-fn entity-type-name expression-renderer-fn group-keyword)}])

(defn- disjoint-property-axioms-as-compartment [ontology id property ontology axiom-getter-fn properties-getter-fn prefix expression-name path-builder-fn]
  (let [axioms (set (axiom-getter-fn ontology property))]
    (set (map (fn [axiom]
                (disjoint-property-axiom-as-compartment ontology id property axiom properties-getter-fn prefix expression-name path-builder-fn))
              axioms))))

(defn- equivalent-property-axioms-as-compartment [ontology id property ontology axiom-getter-fn properties-getter-fn prefix expression-name path-builder-fn]
  (let [axioms (set (axiom-getter-fn ontology property))]
    (set (map (fn [axiom]
                (equivalent-property-axiom-as-compartment ontology id property axiom properties-getter-fn prefix expression-name path-builder-fn))
              axioms))))

(defn- property-description-axioms-as-compartment [ontology id property ontology axiom-getter-fn properties-getter-fn prefix expression-name path-builder-fn]
  (let [axioms (set (axiom-getter-fn ontology property))]
    (set (map (fn [axiom]
                (property-description-axiom-as-compartment ontology id property axiom properties-getter-fn prefix expression-name path-builder-fn))
              axioms))))

(defn- property-annotation-axioms-as-compartment [ontology id property ontology axiom-getter-fn properties-getter-fn prefix expression-name path-builder-fn]
  (let [axioms (set (axiom-getter-fn ontology property))]
    (set (map (fn [axiom]
                (property-annotation-axiom-as-compartment id property axiom properties-getter-fn prefix expression-name path-builder-fn ontology))
              axioms))))

(defn- data-property-annotation-axioms-as-compartment [ontology id property ontology axiom-getter-fn properties-getter-fn prefix expression-name path-builder-fn]
  (let [axioms (set (axiom-getter-fn ontology property))]
    (set (map (fn [axiom]
                (data-property-annotation-axiom-as-compartment id property axiom properties-getter-fn prefix expression-name path-builder-fn ontology))
              axioms))))

(defn- property-chain-axiom-list-as-compartment [ontology id property ontology axiom-getter-fn properties-getter-fn prefix expression-name path-builder-fn]
  (let [axioms (set (axiom-getter-fn ontology property))]
    (set (map (fn [axiom]
                (property-chain-axiom-as-compartment id property axiom properties-getter-fn prefix expression-name path-builder-fn))
              axioms))))

(defn create-object-property-axiom-as-comppartment-exporter [axiom-exporter-fn name axiom-getter-fn properties-getter-fn expression-name]
  {:name name
   :group "object properties"
   :chunker (fn [acc _] (:object-properties acc))
   :exporter (fn [[property inv-property :as p] acc ontology]
               (let [id (entity-id p)]
                 (if (entity-has-rendition-by-id id acc)
                   [(axiom-exporter-fn ontology id property ontology axiom-getter-fn properties-getter-fn "" expression-name
                                       (fn [id] ["Elements" "Association" id :compartments "Role"]))
                    (if inv-property
                      (axiom-exporter-fn ontology id inv-property ontology axiom-getter-fn properties-getter-fn ""
                                         expression-name
                                         (fn [id] ["Elements" "Association" id :compartments "InvRole"])))])))})

(defn create-object-property-characteristic-axiom-as-compartment-exporter [name axiom-getter-fn]
  (create-object-property-axiom-as-comppartment-exporter property-characteristic-axiom-as-compartment
                                                         (str "Is " name)
                                                         axiom-getter-fn
                                                         (fn)
                                                         name))

(defn create-data-property-axiom-as-compartment-exporter [axiom-exporter-fn name axiom-getter-fn properties-getter-fn expression-name]
  {:name name
   :group "data properties"
   :chunker (fn [_ ontology] (set (remove #(= % (get-owl-top-data-prop)) (.getDataPropertiesInSignature ontology))))
   :exporter (fn [property acc ontology]
               (axiom-exporter-fn ontology (entity-id property) property ontology axiom-getter-fn properties-getter-fn "" expression-name
                                  (fn [id] [:compartments id])))})

(defn- cardinality-restrictions-as-map-chunkur
  ([property-type subclass-axioms]
     (if (= property-type :data-prop-restrictions)
       (cardinality-restrictions-as-map-chunkur subclass-axioms OWLDataMinCardinality OWLDataMaxCardinality OWLDataExactCardinality OWLDataSomeValuesFrom)
       (cardinality-restrictions-as-map-chunkur subclass-axioms OWLObjectMinCardinality OWLObjectMaxCardinality OWLObjectExactCardinality OWLObjectSomeValuesFrom)))
  ([subclass-axioms min-type max-type exact-type some-type]
     (reduce (fn [acc subclass-axiom]
	       (let [super (.getSuperClass subclass-axiom)]
		 (if-not (some #(instance? % super) [min-type max-type exact-type some-type])
		   acc
		   (let [subclass (as-class (.getSubClass subclass-axiom))
			 filler (.getFiller super)
			 property (.getProperty super)
			 update-fn (fn [key val-getter-fn]
				     (-> acc
					 (update-in [subclass property filler key] conj (val-getter-fn super))
					 (update-in [subclass property filler :axioms] conj subclass-axiom)))]
		     (cond (instance? min-type super)
			   (update-fn :min (memfn getCardinality))
                           
			   (instance? max-type super)
			   (update-fn :max (memfn getCardinality))
                           
			   (instance? exact-type super)
			   (update-fn :exact (memfn getCardinality))
                           
			   (instance? some-type super)
			   (update-fn :min (fn [_] 1))

			   :else acc)))))
	     {}
	     subclass-axioms)))


(defn- create-object-property-restriction-exporter-helper [name restriction-type keyword value-calculator-fn]
  {:name (str "show subclass of " name " as line")
   :group "object properties"
   :chunker (fn [acc _]
              (set (filter #(instance? restriction-type (.getSuperClass %)) (:subclass-axioms acc #{}))))
   :exporter (fn [subclass-axiom acc _]
               (let [super (.getSuperClass subclass-axiom)
                     subclass (as-class (.getSubClass subclass-axiom))
                     filler (as-class (.getFiller super))
                     property (.getProperty super)
                     subclass-id (entity-id subclass)
                     filler-id (entity-id filler)]
                 (if (and (entity-has-rendition-by-id subclass-id acc) (entity-has-rendition-by-id filler-id acc))
                   {:additions (assoc-in {}
                                         [:object-property-restrictions subclass-id filler-id property]
                                         {keyword (value-calculator-fn super)
                                          :property property
                                          :axioms #{subclass-axiom}})
                    :deletions {:subclass-axioms #{subclass-axiom}}})))})
(defn- create-object-property-restriction-exporter [name restriction-type keyword]
  (create-object-property-restriction-exporter-helper name restriction-type keyword (fn [_] true)))
(defn- create-object-property-quantified-restriction-exporter [name restriction-type keyword]
  (create-object-property-restriction-exporter-helper name restriction-type keyword (memfn getCardinality)))

;
(defn is-rendered-by-name []
	#_(= (.getRendererPluginByClassName (org.protege.editor.owl.ui.renderer.OWLRendererPreferences/getInstance)
																		"org.protege.editor.owl.ui.renderer.OWLEntityRendererImpl")
		 (.getRendererPlugin (org.protege.editor.owl.ui.renderer.OWLRendererPreferences/getInstance))))


(defn get-ontology-prefixes [o]
	(into {} 
			(filter (fn [[prefix iri]]
						(not (some #(.startsWith iri %) default-prefix-list)))
				(map (fn [[prefix iri]]
				 			[(apply str (butlast prefix)) iri])
						(.. OWLOntologyManager (getOntologyFormat o) (asPrefixOWLOntologyFormat) (getPrefixName2PrefixMap))))))

(def export-blocks
  [{:name "always ontology export"
    :chunker (fn [_ ontology] ontology)
    :exporter (fn [ontology _ _]
                {:ontology 
                 {:name (short-namespace-name ontology)
                  :uri (let [ontology-iri (.getOntologyIRI (.getOntologyID ontology))]
                         (if ontology-iri (.toString ontology-iri) ""))
									:read_only_properties (not (is-rendered-by-name))
									:prefixes (get-ontology-prefixes ontology)
									:stats {:ontology_id (str (.getOntologyID ontology))
                          :axiom_count (.getAxiomCount ontology)
                          :logical_axiom_count (.getLogicalAxiomCount ontology)}}})}
   {:name "ontology annotations"
    :group "ontology"
    :chunker (fn [_ ontology] (set (.getAnnotations ontology)))
    :exporter annotation-as-box}
   
   {:name "classes"
    :group "classes"
    :chunker (fn [_ ontology] (set (.getClassesInSignature ontology)))
    :exporter (fn [class _ ontology]
                (named-class-box class ontology))}
   {:name "annotation property definitions"
    :group "ontology"
    :chunker (fn [_ ontology] (set (remove (fn [annotation-property] (has-short-rendering-in-owlgred? (.getIRI annotation-property))) (.getAnnotationPropertiesInSignature ontology))))
    :exporter (fn [annotation-property _ ontology]
                (annotation-property-box annotation-property ontology))}
   {:name "class annotations"
    :group "classes"
    :chunker (fn [_ ontology] (for [individual (.getClassesInSignature ontology),
                                    axiom (.getAnnotations individual ontology)]
                                [individual axiom]))
    :exporter class-annotation-as-compartment}
   {:name "keys"
    :group "classes"
    :chunker (fn [_ ontology]
               (set (ontology-axioms-with-type ontology OWLHasKeyAxiom)))
    :exporter has-key-axiom-as-compartment}
   
   {:name "individuals"
    :group "individuals"
    :chunker (fn [_ ontology] (set (.getIndividualsInSignature ontology)))
    :exporter (fn [individual _ ontology]
                (individual-box individual ontology))}
   {:name "individual annotations"
    :group "individuals"
    :chunker (fn [_ ontology] (for [individual (.getIndividualsInSignature ontology),
                                    axiom (.getAnnotations individual ontology)]
                                [individual axiom]))
    :exporter object-annotation-as-compartment}
   
   {:name "merge inverse object properties"
    :group "object properties"
    :before (fn [_ ontology]
              {:object-properties (set (map vector (remove #(= % (get-owl-top-obj-prop)) (.getObjectPropertiesInSignature ontology))))})
    :chunker (fn [_ ontology] (pairs-of-inverse-object-properties ontology))
    :exporter (fn [property-pair _ _]
                {:additions {:object-properties #{property-pair}}
                 :deletions {:object-properties (set [[(first property-pair)] [(second property-pair)]])}})}
   {:name "object properties"
    :group "object properties"
    :chunker (fn [acc _] (:object-properties acc))
    :exporter (fn [property _ ontology]
                (object-property-line property ontology))}
   (create-object-property-axiom-as-comppartment-exporter property-description-axioms-as-compartment
                                                          "Sub object properties"
                                                          (fn [ontology property] (remove #(= (.getSuperProperty %) (get-owl-top-obj-prop))
                                                                                          (.getObjectSubPropertyAxiomsForSubProperty ontology property)))
                                                          (fn [axiom _] #{(.getSuperProperty axiom)})
                                                          "SuperProperties")
   (create-object-property-axiom-as-comppartment-exporter equivalent-property-axioms-as-compartment
                                                          "Equivalent object properties"
                                                          (fn [ontology property] (.getEquivalentObjectPropertiesAxioms ontology property))
                                                          (fn [axiom property] (disj (set (.getProperties axiom)) property))
                                                          "EquivalentProperties")
   (create-object-property-axiom-as-comppartment-exporter disjoint-property-axioms-as-compartment
                                                          "Disjoint object properties"
                                                          (fn [ontology property] (.getDisjointObjectPropertiesAxioms ontology property))
                                                          (fn [axiom property] (disj (set (.getProperties axiom)) property))
                                                          "DisjointProperties")
   (create-object-property-axiom-as-comppartment-exporter property-chain-axiom-list-as-compartment
                                                          "Property chains"
                                                          (fn [ontology property] 
                                                            (filter #(= property (.getSuperProperty %))
                                                                    (set (ontology-axioms-with-type ontology OWLSubPropertyChainOfAxiom))))
                                                          (fn)
                                                          "PropertyChains")
   (create-object-property-characteristic-axiom-as-compartment-exporter "Functional"
                                                                        (fn [ontology property]
                                                                          (.getFunctionalObjectPropertyAxioms ontology property)))
   (create-object-property-characteristic-axiom-as-compartment-exporter "InverseFunctional"
                                                                        (fn [ontology property]
                                                                          (.getInverseFunctionalObjectPropertyAxioms ontology property)))
   (create-object-property-characteristic-axiom-as-compartment-exporter "Symmetric"
                                                                        (fn [ontology property]
                                                                          (.getSymmetricObjectPropertyAxioms ontology property)))
   (create-object-property-characteristic-axiom-as-compartment-exporter "Asymmetric"
                                                                        (fn [ontology property]
                                                                          (.getAsymmetricObjectPropertyAxioms ontology property)))
   (create-object-property-characteristic-axiom-as-compartment-exporter "Reflexive"
                                                                        (fn [ontology property]
                                                                          (.getReflexiveObjectPropertyAxioms ontology property)))
   (create-object-property-characteristic-axiom-as-compartment-exporter "Irreflexive"
                                                                        (fn [ontology property]
                                                                          (.getIrreflexiveObjectPropertyAxioms ontology property)))
   (create-object-property-characteristic-axiom-as-compartment-exporter "Transitive"
                                                                        (fn [ontology property]
                                                                          (.getTransitiveObjectPropertyAxioms ontology property)))
   (create-object-property-axiom-as-comppartment-exporter property-annotation-axioms-as-compartment
                                                          "object property annotations"
                                                          (fn [ontology property]
                                                            (set (.getAnnotationAssertionAxioms ontology (.getIRI property))))
                                                          (fn)
                                                          "Annotation")
   
   
   {:name "data properties"
    :group "data properties"
    :chunker (fn [_ ontology] (set (remove #(= % (get-owl-top-data-prop)) (.getDataPropertiesInSignature ontology))))
    :exporter (fn [property _ ontology]
                (data-property-compartment property ontology))}
   (create-data-property-axiom-as-compartment-exporter property-description-axioms-as-compartment
                                                       "Sub data properties"
                                                       (fn [ontology property] (remove #(= (.getSuperProperty %) (get-owl-top-data-prop))
                                                                                       (.getDataSubPropertyAxiomsForSubProperty ontology property)))
                                                       (fn [axiom _] #{(.getSuperProperty axiom)})
                                                       "SuperProperties")
   (create-data-property-axiom-as-compartment-exporter property-description-axioms-as-compartment
                                                       "Equivalent data properties"
                                                       (fn [ontology property] (.getEquivalentDataPropertiesAxioms ontology property))
                                                       (fn [axiom property] (disj (set (.getProperties axiom)) property))
                                                       "EquivalentProperties")
   (create-data-property-axiom-as-compartment-exporter property-description-axioms-as-compartment
                                                       "Disjoint data properties"
                                                       (fn [ontology property] (.getDisjointDataPropertiesAxioms ontology property))
                                                       (fn [axiom property] (disj (set (.getProperties axiom)) property))
                                                       "DisjointProperties")
   (create-data-property-axiom-as-compartment-exporter property-characteristic-axiom-as-compartment
                                                       "Is Functional"
                                                       (fn [ontology property] (.getFunctionalDataPropertyAxioms ontology property))
                                                       (fn)
                                                       "IsFunctional")
   (create-data-property-axiom-as-compartment-exporter data-property-annotation-axioms-as-compartment
                                                       "data property annotations"
                                                       (fn [ontology property]
                                                         (set (.getAnnotationAssertionAxioms ontology (.getIRI property))))
                                                       (fn)
                                                       "ASFictitiousAnnotation")

   (create-group-exporter {:name "disjoint classes"
                           :export-group-name "classes"
                           :axiom-type OWLDisjointClassesAxiom
                           :getter-fn (memfn getClassExpressions)
                           :group-keyword :groups-of-disjoint-classes
                           :group-box-type-name  "DisjointClasses"
                           :entity-getter-fn as-class
                           :box-label "disjoint"
                           :connection-line-type-name  "Connector"
                           :entity-renderer-fn class-box
                           :group-line-type-name "Disjoint"
                           :line-label "disjoint"
                           :entity-type-name "Class"
                           :expression-renderer-fn #(expressions-as-5-deep-text %1 %2 "DisjointClass")})
   (create-group-exporter {:name "equivalent classes"
                           :export-group-name "classes"
                           :axiom-type OWLEquivalentClassesAxiom
                           :getter-fn (memfn getClassExpressions)
                           :group-keyword :groups-of-equivalent-classes
                           :group-box-type-name  "EquivalentClasses"
                           :entity-getter-fn as-class
                           :box-label "equivalent"
                           :connection-line-type-name  "Connector"
                           :entity-renderer-fn class-box
                           :group-line-type-name "EquivalentClass"
                           :line-label "equivalent"
                           :entity-type-name "Class"
                           :expression-renderer-fn #(expressions-as-5-deep-text %1 %2 "EquivalentClass")})
   (create-group-exporter {:name "different individuals"
                           :export-group-name "individuals"
                           :axiom-type OWLDifferentIndividualsAxiom
                           :getter-fn (memfn getIndividuals)
                           :group-keyword :groups-of-different-individuals
                           :group-box-type-name  "DifferentIndivids"
                           :entity-getter-fn id
                           :box-label "different"
                           :connection-line-type-name  "Connector"
                           :entity-renderer-fn individual-box
                           :group-line-type-name "DifferentIndivid"
                           :line-label "different"
                           :entity-type-name "Object"
                           :expression-renderer-fn #(expressions-newline-separated-text %1 "Different individuals")})
   (create-group-exporter {:name "same individuals"
                           :export-group-name "individuals"
                           :axiom-type  OWLSameIndividualAxiom
                           :getter-fn (memfn getIndividuals)
                           :group-keyword :groups-of-same-individuals
                           :group-box-type-name  "SameAsIndivids"
                           :entity-getter-fn id
                           :box-label "sameAs"
                           :connection-line-type-name  "Connector"
                           :entity-renderer-fn individual-box
                           :group-line-type-name "SameAsIndivid"
                           :line-label "sameAs"
                           :entity-type-name "Object"
                           :expression-renderer-fn #(expressions-newline-separated-text %1 "Same individuals")})
   {:name "class assertions"
    :group "individuals"
    :chunker (fn [_ ontology]
               (set (ontology-axioms-with-type ontology OWLClassAssertionAxiom)))
    :exporter class-assertion-line}
   {:name "object property assertions"
    :group "individuals"
    :chunker (fn [_ ontology]
               (set (ontology-axioms-with-type ontology OWLObjectPropertyAssertionAxiom)))
    :exporter positive-object-property-assertion-line}
   {:name "data property assertions"
    :group "individuals"
    :chunker (fn [_ ontology]
               (set (ontology-axioms-with-type ontology OWLDataPropertyAssertionAxiom)))
    :exporter positive-data-property-assertion}
   {:name "negative object property assertions"
    :group "individuals"
    :chunker (fn [_ ontology]
               (set (ontology-axioms-with-type ontology OWLNegativeObjectPropertyAssertionAxiom)))
    :exporter negative-object-property-assertion-line}
   {:name "negative data property assertions"
    :group "individuals"
    :chunker (fn [_ ontology]
               (set (ontology-axioms-with-type ontology OWLNegativeDataPropertyAssertionAxiom)))
    :exporter negative-data-property-assertion}

   {:name "subclass axioms"
    :chunker (fn [_ ontology]
               (set  (ontology-axioms-with-type ontology OWLSubClassOfAxiom)))
    :exporter (fn [axiom _ _] {:subclass-axioms #{axiom}})}

   {:name (str "show subclass of cardinality restrictions as multiplicity")
    :group "object properties"
    :chunker (fn [accumulator ontology]
               (let [card-restriction-map (cardinality-restrictions-as-map-chunkur :object-prop-restrictions (:subclass-axioms accumulator #{}))]
                  (for [[subclass properties] card-restriction-map,
                        [property targets] properties,
                        [target multiplicity] targets]
                    [subclass property target multiplicity])))
    :exporter (fn [[subclass property target multiplicity] accumulator ontology]
                (let [subclass (as-class subclass)
                      filler (as-class target)
                      subclass-id (entity-id subclass)
                      filler-id (entity-id filler)
                      property-id (entity-id property)
                      ontology-imports-closure (.getImportsClosure ontology)
                      domain (as-class (.getDomains property ontology-imports-closure))
                      range (as-class (.getRanges property ontology-imports-closure))
                      axioms (set (:axioms multiplicity))]
                  (if (and (entity-has-rendition-by-id subclass-id accumulator)
                           (entity-has-rendition-by-id filler-id accumulator)
                           (entity-has-rendition-by-id (get (:mappings accumulator) property-id) accumulator)
                           (= domain subclass)
                           (= range filler))
                    (let [id (entity-id multiplicity)]
                      {:additions {:compartments
                                   {id
                                    {(if (object-property-redered-as-inverse? accumulator property)
                                       "InvRole"
                                       "Role")
                                     {"Multiplicity" (multiplicity-as-text multiplicity)}}}
                                   
                                   :compartment-parents
                                   {id {:parent-id (get (:mappings accumulator) (entity-id property))
                                        :parent-type "Association"}}
                                   
                                   :exported-axioms axioms}
                       :deletions {:subclass-axioms axioms}}))))}

   {:name (str "show subclass of cardinality restrictions as multiplicity")
    :group "data properties"
    :chunker (fn [accumulator ontology]
               (let [card-restriction-map (cardinality-restrictions-as-map-chunkur :data-prop-restrictions (:subclass-axioms accumulator #{}))]
                 (for [[subclass properties] card-restriction-map,
                       [property targets] properties,
                       [target multiplicity] targets]
                   [subclass property target multiplicity])))
    :exporter (fn [[subclass property target multiplicity] accumulator ontology]
                (let [subclass (as-class subclass)
                      subclass-id (entity-id subclass)
                      property-id (entity-id property)
                      ontology-imports-closure (.getImportsClosure ontology)
                      domain (as-class (.getDomains property ontology-imports-closure))
                      range (set (.getRanges property ontology-imports-closure))
                      axioms (set (:axioms multiplicity))]
                  (if (and (entity-has-rendition-by-id subclass-id accumulator)
                           (= domain subclass)
                           (= range #{target}))
                    (let [id (entity-id multiplicity)]
                      {:additions {:compartments
                                   {property-id
                                    {"Multiplicity" (multiplicity-as-text multiplicity)}}
                                   :exported-axioms axioms}
                       :deletions {:subclass-axioms axioms}}))))}

   (create-object-property-restriction-exporter "some" OWLObjectSomeValuesFrom :some)
   (create-object-property-restriction-exporter "only" OWLObjectAllValuesFrom :all)
   (create-object-property-quantified-restriction-exporter "min" OWLObjectMinCardinality :min)
   (create-object-property-quantified-restriction-exporter "max" OWLObjectMaxCardinality :max)
   (create-object-property-quantified-restriction-exporter "exact" OWLObjectExactCardinality :exact)
   
   {:name "object property restrictions as lines"
    :group "object properties"
    :chunker (fn [acc _]
               (reduce (fn [a [k1 r1]]
                         (apply conj
                                a
                                (reduce (fn [a [k2 r2]]
                                          (apply conj
                                                 a
                                                 (reduce (fn [a [k3 r3]]
                                                           (conj a [k1 k2 k3 r3]))
                                                         []
                                                         r2)))
                                        []
                                        r1)))
                       []
                       (:object-property-restrictions acc)))
    :exporter object-property-restrinction-as-line}
   
   {:name "anonymous subclasses as boxes"
    :group "classes"
    :chunker (fn [acc _] (set (filter #(not (isa? (type %) OWLClass)) (map (memfn getSubClass) (:subclass-axioms acc)))))
    :exporter (fn [class _ ontology]
                (class-box class ontology))}
   {:name "make top classes subclass of Thing"
    :group "classes"
    :chunker (fn [acc ontology]
               (disj
                (set
                 (filter
                  #(not
                    (some (fn [cl]
                            (entity-has-rendition-by-id (entity-id cl) acc))
                          (concat (.getSuperClasses % ontology)
                                 (flatten
                                  (map (fn [union-expression]
                                         (seq (.getOperands union-expression)))
                                       (filter (fn [expression]
                                                 (instance? OWLObjectIntersectionOf expression))
                                               (.getEquivalentClasses % ontology)))))))
                  (.getClassesInSignature ontology)))
                (get-owl-thing)))
    :exporter (fn [class _ _]
                {:subclass-axioms #{(.getOWLSubClassOfAxiom (.getOWLDataFactory OWLOntologyManager) class (get-owl-thing))}})
    :after (fn [_ ontology]
             (named-class-box (get-owl-thing) ontology))}
   {:name "make named classes subclass of corresponding anonymous unions"
    :group "classes"
    :chunker (fn [acc ontology]
               (let [expressions-with-properties (filter 
                                                  #(isa? (type %)
                                                         OWLObjectUnionOf)
                                                  (apply concat 
                                                         (map #(concat (.getDomains % ontology)
                                                                       (.getRanges % ontology))
                                                              (concat (.getDataPropertiesInSignature ontology)
                                                                      (.getObjectPropertiesInSignature ontology)))))]
                 (for [exp expressions-with-properties, cl (.getOperands exp)] [cl exp])))
    :exporter (fn [[sub-class union-class] _ _]
                {:subclass-axioms #{(.getOWLSubClassOfAxiom (.getOWLDataFactory OWLOntologyManager) sub-class union-class)}})
    :after (fn [_ ontology]
             (named-class-box (get-owl-thing) ontology))}
   {:name "merge subclass lines with forks"
    :group "classes"
    :chunker (fn [acc _] (set (filter (fn [class]
                                        (entity-has-rendition-by-id (entity-id (as-class class)) acc))
                                      (map (memfn getSuperClass) (:subclass-axioms acc)))))
    :exporter (fn [class acc _]
                (let [subclass-axioms (set (filter (fn [axiom]
                                                     (= (.getSuperClass axiom) class))
                                                   (:subclass-axioms acc)))]
                  (if (> (count subclass-axioms) 1)
                    {:deletions {:subclass-axioms subclass-axioms}
                     :additions {:subclass-forks {class [(map (memfn getSubClass) subclass-axioms) subclass-axioms]}}})))}

   {:name "subclass forks"
    :chunker (fn [acc _] (:subclass-forks acc))
    :exporter generalisation-fork}
   {:name "subclass lines"
    :group "classes"
    :chunker (fn [acc _] (:subclass-axioms acc))
    :exporter generalisation-line}
   {:name "other super class as text"
    :group "classes"
    :chunker (fn [acc _] (:subclass-axioms acc))
    :exporter generalisation-compartment}

   {:name "equivalent to 'and' named class as subclass link"
    :group "classes"
    :chunker (fn [_ ontology]
               (set (ontology-axioms-with-type ontology OWLEquivalentClassesAxiom)))
    :exporter (fn [equivalent-class-axiom acc _]
                (let [descriptions (seq (.getClassExpressions equivalent-class-axiom))
                      intersections (filter (fn [description] (isa? (type description) OWLObjectIntersectionOf)) descriptions)
                      operands (flatten (map (fn [intersection] (seq (.getOperands intersection))) intersections))
                      operands-as-classes (map as-class operands)
                      operands-as-classes-with-rendition (filter (fn [operand-as-class]
                                                                   (entity-has-rendition-by-id (entity-id operand-as-class) acc))
                                                                 operands-as-classes)
                      descriptions-as-classes-with-rendition (filter (fn [description-as-class]
                                                                       (entity-has-rendition-by-id (entity-id description-as-class) acc))
                                                                     (map as-class descriptions))]
                  (set (map (fn [description-as-class]
                              (set (map (fn [operand]
                                          (if-not 
                                              (subclass-relation-rendered-graphicaly? acc 
                                                                                      (entity-id description-as-class)
                                                                                      (entity-id (as-class operand)))
                                            (assoc-in {}
                                                      ["Elements" "Generalization" (entity-id [description-as-class operand])]
                                                      {:source (entity-id description-as-class)
                                                       :target (entity-id (as-class operand))})))
                                        operands-as-classes-with-rendition)))
                            descriptions-as-classes-with-rendition))))}
   
   {:name "add compartments to elements"
    :chunker (fn [acc _] (:compartments acc))
    :exporter (fn [[compartment-id value] acc _]
                (let [{:keys [parent-id parent-type path super-compartment] :as parent-info} (get (:compartment-parents acc) compartment-id)]
                  {:additions (assoc-in {} (concat ["Elements" parent-type parent-id :compartments] path)
                                        (if super-compartment
                                          #{{super-compartment value}}
                                          value))}))}

   {:name "data types"
    :group "data-type"
    :chunker (fn [_ ontology]
               (set (filter #(not (.isBuiltIn %)) (.getDatatypesInSignature ontology))))
    :exporter data-type-box}
   
   {:name "unexported axioms"
    :group "ontology"
    :before (fn [acc ontology]
              (let [exported-axioms (get acc :exported-axioms #{})
                    unexported (difference (set (.getAxioms ontology)) exported-axioms)]
                (set (map (fn [axiom] {:unexported #{(render axiom)}})
                          unexported))))}])

(def export-groups [])

(defn ontology-imports-closure [ontology]
  (add-ontology-imports-closure! (atom {}) ontology))

(defn ontology-dependencies [ontology]
  (reduce (fn [acc [ontology-id ontology]]
            (assoc acc ontology-id (map #(entity-id %)
                                        (.getDirectImports ontology))))
          {}
          (ontology-imports-closure ontology)))

(defn render-ontology-imports [ontology]
  (let [dependencies (ontology-dependencies ontology)]
    {"Import" (reduce (fn [acc [onto-id import-id :as p]]
                        (assoc acc (entity-id p) {:source onto-id
                                                  :target import-id
                                                  :compartments {"NewCompartment" "<<import>>"}}))
                      {}
                      (reduce (fn [acc [onto-id import-ids]]
                                (into acc (map (fn [import-id] [onto-id import-id]) import-ids)))
                              []
                              dependencies))}))

(defn- get-export-operation-count [export-blocks]
  (loop [n 0
         block (first export-blocks)
         others (rest export-blocks)]
    (cond
     (nil? block) n
     (vector? block) (recur (+ n (get-export-operation-count block)) (first others) (rest others))
     :else (recur (inc n) (first others) (rest others)))))

(defn estimate-export-work-amount [ontology]
  (* (get-export-operation-count export-blocks)
     (count (ontology-imports-closure ontology))))

(defn set-render-plugin-to-render-by-name)

(defn data-for-export
  ([ontology]
     (export-with ontology export-blocks))
  ([ontology canceled? done? progress]
     (export-with ontology export-blocks canceled? done? progress)))

(defn relevant-data-for-export [ontology canceled? done? progress]
  (dissoc-all-except (data-for-export ontology canceled? done? progress) ["Elements" :ontology :unexported]))


(defn ontology-with-imports-closure-for-export [ontology canceled? done? progress]
  (try
    (do (reset-id-map)
        (let [ontology-imports-closure (ontology-imports-closure ontology)
              export-data {:active-ontology (entity-id ontology)
                           "Elements" (render-ontology-imports ontology)
                           :ontologies (zipmap (keys ontology-imports-closure)
                                               (map #(relevant-data-for-export % canceled? done? progress)
                                                    (vals ontology-imports-closure)))}]
          export-data))
    (catch java.lang.Exception e (do (println (pst-str e))
                                     (reset! canceled? true)
                                     (show-error-message "Error while exporting"
                                                         (pst-str e))))))


(defn- show-error-message-problem-connecting-to-owlgred []
  (show-error-message "Cannot connect to OWLGrEd"
                      "Cannot connect to OWLGrEd!\nMake sure that\n - OWLGrEd is running and\n - some project is open"))


(defn ontology-to-tda [ontology canceled? done? progress finished-data-ref]
	(let [data (ontology-with-imports-closure-for-export ontology canceled? done? progress)]
	  (if-not @canceled?
			(dosync (alter finished-data-ref merge
	                               {:finished true
																	:data data}))))
  (reset! done? true))

(defn export-to-tda-with-progress-monitor* [ontology]
   (process-with-progress-monitor* ontology
     ontology-to-tda
     (estimate-export-work-amount ontology)))



                                        ;for testing
;(def o (.getActiveOntology OWLModelManager))
;(use 'clojure.contrib.pprint)
;(use 'clj-stacktrace.repl)

(defn ontology-from-file [path]
  ; (throw (Exception. (str (org.semanticweb.owlapi.util.VersionInfo/getVersionInfo))))
  (let [file (new File path)
        source (new FileDocumentSource file)
        config (.setMissingImportHandlingStrategy (new OWLOntologyLoaderConfiguration)
                                                  MissingImportHandlingStrategy/SILENT)]
    (.loadOntologyFromOntologyDocument OWLOntologyManager source config)))

(defn ontology-to-string [ontology]
	(intern 'lv.lumii.tda '*ontology* ontology)
	;(println "-----\n" ontology "\n---------")
	(let [data-ref (export-to-tda-with-progress-monitor* ontology)
				data (:data @data-ref)
				serialized-form (serialize data)]
		;(println "------" data)
		serialized-form))


(defn embeded-export-to-tda [{ontology-path "ontology_path" preferences "preferences" {:strs [prefixes axioms]} "custom_render_spec" :as table}]
	; (println "******* full export ********")
	(swap! lv.lumii.tda/*custom-render-spec-ontology-atom* 
				 (fn [_] (load-custom-render-pref-ontology prefixes axioms)))
  (swap! lv.lumii.tda/*preferences* 
         (fn [_] (or preferences {})))
	(let [ontology (ontology-from-file ontology-path)]
		(ontology-to-string ontology)))
		
;
(defn embeded-module-export-to-tda [{ontology-path "ontology_path" signature "signature" preferences "preferences" {:strs [prefixes axioms]} "custom_render_spec" :as table}]
	; (println "******* module export ********")
	(swap! lv.lumii.tda/*custom-render-spec-ontology-atom* 
				 (fn [_] (load-custom-render-pref-ontology prefixes axioms)))
  (swap! lv.lumii.tda/*preferences* 
         (fn [_] (or preferences {})))
	(let [ontology (ontology-from-file ontology-path)
				ontology-entities (.getSignature ontology)
				
				extractor (SyntacticLocalityModuleExtractor. OWLOntologyManager ontology ModuleType/TOP)
				
				signature (set signature)
				;a (println signature)
				
				module-signature (set (seq (filter #(some #{(.. % (getIRI) (getFragment))} signature) ontology-entities)))
				;b (println module-signature)
				
				iri (.. ontology (getOntologyID) (getOntologyIRI))
				
				ontology-for-export (do (.removeOntology OWLOntologyManager ontology) ;remove current ontology, so that we can reuse iri
																(.extractAsOntology extractor module-signature iri))]
		(ontology-to-string ontology-for-export)))
		

;
(defn create-tmp-file-write-ontology-and-map-to-iri! [OWLOntologyManager ontology-iri ontology-str]
	(let [tmp-file (. File createTempFile "tmp-ontology" ".owl")
				out-stream (new OutputStreamWriter (new FileOutputStream tmp-file) "UTF8")]
		; (println tmp-file)
		(.deleteOnExit tmp-file)
    (.write out-stream ontology-str)
    (.close out-stream)
		; (println ontology-str)
		; add iri mapping, so that imports can find it later
		(.addIRIMapper OWLOntologyManager (org.semanticweb.owlapi.util.SimpleIRIMapper. (IRI/create ontology-iri) (IRI/create tmp-file)))))

(defn embeded-export-to-tda-from-pratege [{ontologies "ontologies" preferences "preferences" active-ontology-iri "active_ontology_iri" {:strs [prefixes axioms]} "custom_render_spec" :as table}]
	;(println "******* protege export ********")
	; (println ontologies)
  ; (println preferences)
	(swap! lv.lumii.tda/*custom-render-spec-ontology-atom* 
				 (fn [_] (load-custom-render-pref-ontology prefixes axioms)))
  (swap! lv.lumii.tda/*preferences* 
         (fn [_] (or preferences {})))
	(.clearIRIMappers OWLOntologyManager)
	(doseq [{:strs [ontology-iri ontology-str]} ontologies]
		(create-tmp-file-write-ontology-and-map-to-iri! OWLOntologyManager ontology-iri ontology-str))
	(ontology-to-string (.loadOntology OWLOntologyManager (IRI/create active-ontology-iri))))