(ns
    #^{:author "renarl",
       :doc "renderers to tda form"}
  lv.lumii.tda.export.renderer
  (:use lv.lumii.tda
        lv.lumii.tda.utilities
        lv.lumii.tda.export.utilities
        clojure.set
        [clojure.string :only (split)])
  (:import [org.semanticweb.owlapi.model OWLClass OWLIndividual OWLDataProperty OWLObjectProperty OWLLiteral IRI]
           [uk.ac.manchester.cs.owl.owlapi OWLDataIntersectionOfImpl]
           [uk.ac.manchester.cs.owl.owlapi.mansyntaxrenderer ManchesterOWLSyntaxObjectRenderer ManchesterOWLSyntaxPrefixNameShortFormProvider]
           [java.io StringWriter]))

(defn render-to-manchester-form [ontology expression]
  (let [short-form-provider  (proxy [ManchesterOWLSyntaxPrefixNameShortFormProvider] [OWLOntologyManager ontology]
                                (getShortForm [owl-entity]
                                   (let [{:keys [rendering name namespace iri]} (get-entity-rendering-name-namespace ontology owl-entity)]
                                      (if (or rendering (has-short-rendering-in-owlgred? iri))
                                        (str name)
                                        (str name
                                            "{" namespace "}")))))
        string-writer (new StringWriter)
        renderer (new ManchesterOWLSyntaxObjectRenderer string-writer short-form-provider)]
    (.visit renderer expression)
    (.toString string-writer)))

(defn expressions-as-3-deep-text [ontology expression expression-name]
  (let [expression-list (if (coll? expression) expression [expression])]
    (assoc-in {}
              [(str "ASFictitious" expression-name)]
              (set (map (fn [expression] {expression-name {"Expression" (render-with-namespace-in-curly ontology expression)}}) expression-list)))))


(defn expressions-as-5-deep-text [ontology expression expression-name]
  (let [expression-list (if (coll? expression) expression [expression])]
    (assoc-in {}
              [(str "ASFictitious" expression-name "es")]
              (hash-set (assoc-in {}
                                  [(str expression-name "es")
                                   (str "ASFictitious" expression-name)]
                                  (set (map (fn [expression] 
                                              {expression-name 
                                               {"Expression" 
                                                (render-with-namespace-in-curly ontology expression)}})
                                            expression-list)))))))

(defn expressions-newline-separated-text [ontology expression-list expression-name]
  (assoc-in {}
            [expression-name]
            #{(concat-with (map #(render-with-namespace-in-curly ontology %1) expression-list) "\n")}))

(defn anonymous-class-box [expression-list ontology]
  (let [id (entity-id (as-class expression-list))]
    {"Elements"
     {"Class"
      {id {:compartments
           (expressions-as-5-deep-text ontology expression-list "EquivalentClass")}}}
     :rendered-entities #{id}}))

(defn- name-with-namespace [ontology entity name-subcompartment-id namespace-subcompartment-id]
  (let [{:keys [rendering name namespace]} (get-entity-rendering-name-namespace ontology entity)]
    (if rendering
      {name-subcompartment-id name
       namespace-subcompartment-id namespace}
      {name-subcompartment-id name
       namespace-subcompartment-id namespace})))

(defn named-class-box [class ontology]
  (let [id (entity-id (as-class class))]
    [{"Elements"
      {"Class"
       {id {:compartments
            {"Name" (name-with-namespace ontology class "Name" "Namespace")}}}}
      :rendered-entities #{id}}
     (declaration-axioms class ontology)]))

(defn class-box [class ontology]
  (if (instance? OWLClass class)
    (named-class-box class ontology)
    (anonymous-class-box class ontology)))

(defn individual-box [individual ontology]
  (let [id (entity-id individual)]
    [{"Elements"
      {"Object"
       {id
        {:compartments
         {"Title"
          {"Name" (if (.isAnonymous individual)
                    {"Name" (str (.getID individual))}
                    (name-with-namespace ontology individual "Name" "Namespace"))
           "Kols" ":"}}}}}
      :rendered-entities #{id}}
     (if-not (.isAnonymous individual )
       (declaration-axioms individual ontology))]))

(defn object-property-line [[property inv-property :as p] ontology]
  (let [id (entity-id p)
                                        ;assume that range and domain are invreses for inverse
        domain (as-class (get-domain-of-property-pair p ontology))
        range (as-class (get-range-of-property-pair p ontology))]
    [{:mappings {(entity-id property) id}}
     (if inv-property
       [{:mappings {(entity-id inv-property) id}}
        (declaration-axioms inv-property ontology)
        {:exported-axioms (set (.getObjectPropertyDomainAxioms ontology inv-property))}
        {:exported-axioms (set (.getObjectPropertyRangeAxioms ontology inv-property))}
        {:exported-axioms (set (.getInverseObjectPropertyAxioms ontology inv-property))}])
     [{"Elements"
       {"Association"
        {id {:compartments {"Role"    {"Name" (name-with-namespace ontology property "Name" "Namespace")}
                            "InvRole" {"Name" (if inv-property (name-with-namespace ontology inv-property "Name" "Namespace") "")}}
             :source (entity-id domain)
             :target (entity-id range)}}}}
      (declaration-axioms property ontology)
      (class-box domain ontology)
      (class-box range ontology)
      {:exported-axioms (set (.getObjectPropertyDomainAxioms ontology property))}
      {:exported-axioms (set (.getObjectPropertyRangeAxioms ontology property))}]
     {:rendered-entities #{id}}]))

(defn property-characteristic-axiom-as-compartment [ontology id property ontology axiom-getter-fn _ prefix expression-name path-builder-fn]
  (let [axioms (set (axiom-getter-fn ontology property))]
    (if-not (empty? axioms)
      [(assoc-in {}
                 (concat (path-builder-fn id) [(str prefix expression-name)])
                 true)
       {:exported-axioms axioms}])))

(defn equivalent-property-axiom-as-compartment [ontology id property axiom properties-getter-fn prefix expression-name path-builder-fn]
	(let [property-expression-list (properties-getter-fn axiom property)]
	   [(assoc-in {} (path-builder-fn id)
	     	{"EquivalentProperties"
	    			{"ASFictitiousEquivalentProperties"
	   				#{{"EquivalentProperties"
	   			 	  	{"ASFictitiousEquivalentProperty"
	   				 			(seq (map (fn [prop]
	   											{"EquivalentProperty" {"Expression" (render-with-namespace-in-curly ontology prop)}})
	   									  property-expression-list))}}}}})
	    {:exported-axioms #{axiom}}]))

(defn disjoint-property-axiom-as-compartment [ontology id property axiom properties-getter-fn prefix expression-name path-builder-fn]
	(let [property-expression-list (properties-getter-fn axiom property)]
	   [(assoc-in {} (path-builder-fn id)
	     	{"DisjointProperties"
	    			{"ASFictitiousDisjointProperties"
	   				#{{"DisjointProperties"
	   			 	  	{"ASFictitiousDisjointProperty"
	   				 			(seq (map (fn [prop]
	   											{"DisjointProperty" {"Expression" (render-with-namespace-in-curly ontology prop)}})
	   									  property-expression-list))}}}}})
	    {:exported-axioms #{axiom}}]))

(defn property-description-axiom-as-compartment [ontology id property axiom properties-getter-fn prefix expression-name path-builder-fn]
  (let [properties (properties-getter-fn axiom property) ]
    [(assoc-in {}
               (concat (path-builder-fn id) [(str prefix expression-name) (str "ASFictitious" expression-name)])
               (set (map (fn [prop] {expression-name {"Expression" (render-with-namespace-in-curly ontology prop)}}) properties)))
     {:exported-axioms #{axiom}}]))

(defn data-property-compartment [property ontology]
  (let [id (entity-id property)
        ontology-imports-closure (.getImportsClosure ontology)
        domain (as-class (.getDomains property ontology-imports-closure))
        r (.getRanges property ontology-imports-closure)
        range (concat-with (map render r) " and ")]
    [(class-box domain ontology)
     {:exported-axioms (set (.getDataPropertyDomainAxioms ontology property))}
     {:exported-axioms (set (.getDataPropertyRangeAxioms ontology property))}
     (declaration-axioms property ontology)
     {:compartments
      {id {"Name" (name-with-namespace ontology property "Name" "Namespace")
           "Type" (let [rendering (concat-with (map #(render-to-manchester-form ontology %) r) " and ")]
                    (if (and (= 1 (count r))
                             (or (instance? org.semanticweb.owlapi.model.OWLDatatype (first r))
                                 (.startsWith rendering "(")))
                      rendering
                      (if (= range "")
                        ""
                        (str "(" rendering ")"))))}}}
     {:compartment-parents
      {id {:parent-id (entity-id domain)
           :parent-type "Class"
           :path ["ASFictitiousAttributes"]
           :super-compartment "Attributes"}}}]))

(defn group-box-exporter [entity-getter-fn group-box-type-name label connection-line-type-name group-keyword entity-renderer-fn]
  (fn [group acc ontology]
    (let [id (entity-id group)
          entities (map entity-getter-fn group)
          ids (map entity-id entities)]
      {:additions
       [{"Elements"
         {group-box-type-name
          {id {:compartments {"Label" label}
               :outgoing {connection-line-type-name ids}}}}}
        {:exported-axioms (get-in acc [(axiom-keyword-for-group group-keyword) group])}
        (set (map (fn [entity] (entity-renderer-fn entity ontology)) entities))]
       :deletions {group-keyword #{group}}})))

(defn group-line-exporter [entity-getter-fn group-line-type-name label group-keyword]
  (fn [group acc ontology]
    (let [id (entity-id group)
          entities (map entity-getter-fn group)
          ids (map entity-id entities)
          id-pairs (get-all-pairs ids)]
      (if-not (some #(not (entity-has-rendition-by-id %1 acc)) ids) ;render as lines only if all entities have renderings
        {:additions
         (set (map (fn [[cl1-id cl2-id]]
                     [{"Elements"
                       {group-line-type-name
                        {id {:compartments {"Label" label}
                             :source cl1-id
                             :target cl2-id}}}}
                      {:exported-axioms (get-in acc [(axiom-keyword-for-group group-keyword) group])}])
                   id-pairs))
         :deletions {group-keyword #{group}}}))))

                                        ;todo render on single line if more then 2 in group
(defn group-compartment-exporter [entity-getter-fn entity-type-name expressions-renderer-fn group-keyword]
  (fn [group acc ontology]
    {:additions
     (set (map (fn [entity]
                 (let [id (entity-id (entity-getter-fn entity))]
                   (if (entity-has-rendition-by-id id acc)
                     [{"Elements"
                       {entity-type-name
                        {id {:compartments
                             (expressions-renderer-fn ontology (disj (set group) entity))}}}}
                      {:exported-axioms (get-in acc [(axiom-keyword-for-group group-keyword) group])}])))
               group))
     :deletions {group-keyword #{group}}}))

(defn class-assertion-line [class-assertion-axiom _ ontology]
  (let [individual (.getIndividual class-assertion-axiom)
        class (as-class (.getClassExpression class-assertion-axiom))
        individual-id (entity-id individual)
        id (entity-id class-assertion-axiom)]
    [{:exported-axioms #{class-assertion-axiom}}
     {"Elements"
      {"Dependency"
       {id {
            :compartments {"Label" "instanceOf"}
            :source individual-id
            :target (entity-id class)}}}}
     (class-box class ontology)
     (individual-box individual ontology)]))

(defn- object-property-assertion-line [ontology object-property-assertion-axiom negative?]
  (let [subject (.getSubject object-property-assertion-axiom)
        object (.getObject object-property-assertion-axiom)
        property (.getProperty object-property-assertion-axiom)
        subject-id (entity-id subject)
        object-id (entity-id object)
        id (entity-id object-property-assertion-axiom)]
    [{:exported-axioms #{object-property-assertion-axiom}}
     {"Elements"
      {"Link"
       {id {:compartments {"Direct" (assoc (name-with-namespace ontology property "Property" "Namespace")
                                     "IsNegativeAssertion" negative?)}
            :source subject-id
            :target object-id}}}}]))

(defn positive-object-property-assertion-line [object-property-assertion-axiom _ ontology]
  (object-property-assertion-line ontology object-property-assertion-axiom false))

(defn negative-object-property-assertion-line [object-property-assertion-axiom _ ontology]
  (object-property-assertion-line ontology object-property-assertion-axiom true))

(defn- render-owl-literal [ontology literal]
  (let [data-type (.getDatatype literal)]
    (cond 
     (.isString data-type) {:value (str "\"" (.getLiteral literal) "\"")}
     
     (or (.isDouble data-type)
         (.isFloat data-type)
         (.isInteger data-type)) {:value (.getLiteral literal)}
         
         :else {:value (str "\"" (.getLiteral literal) "\"")
                :type (render-with-namespace-in-curly ontology data-type)})))

(defn- data-property-assertion-helper [ontology data-property-assertion-axiom negative?]
  (let [subject (.getSubject data-property-assertion-axiom)
        object (.getObject data-property-assertion-axiom)
        property (.getProperty data-property-assertion-axiom)
        subject-id (entity-id subject)
        id (entity-id data-property-assertion-axiom)
        compart-type-name (if negative? "NegativeDataPropertyAssertion" "DataPropertyAssertion")
        literal-rendition (if (isa? (type object) OWLLiteral)
                            (render-owl-literal ontology object)
                            {:value (render-with-namespace-in-curly ontology object)})]
    {:exported-axioms #{data-property-assertion-axiom}
     :compartments {id {"Property" (:name (get-entity-rendering-name-namespace ontology property))
                        "Value" (:value literal-rendition)
                        "Type" (:type literal-rendition)}}
     :compartment-parents {id {:parent-id subject-id
                               :parent-type "Object"
                               :path [(str "ASFictitious" compart-type-name)]
                               :super-compartment compart-type-name}}}))

(defn positive-data-property-assertion [data-property-assertion-axiom _ ontology]
  (data-property-assertion-helper ontology data-property-assertion-axiom false))

(defn negative-data-property-assertion [data-property-assertion-axiom _ ontology]
  (data-property-assertion-helper ontology data-property-assertion-axiom true))

(defn generalisation-fork [[super [subclasses axioms] :as fork] _ _]
  (let [id (entity-id fork)
        super-id (entity-id (as-class super))
        sub-ids (set (map #(entity-id (as-class %)) subclasses))]
    {"Elements"
     {"HorizontalFork"
      {id {:outgoing {"GeneralizationToFork" super-id}
           :incomming {"AssocToFork" sub-ids}}}}
     :exported-axioms axioms}))

(defn generalisation-line [subclass-axiom acc _]
  (let [id (entity-id subclass-axiom)
        super-id (entity-id (as-class (.getSuperClass subclass-axiom)))
        sub-id (entity-id (as-class (.getSubClass subclass-axiom)))]
    (if (and (entity-has-rendition-by-id super-id acc) (entity-has-rendition-by-id sub-id acc))
      {:additions {"Elements"
                   {"Generalization"
                    {id {:source sub-id
                         :target super-id}}}
                   :exported-axioms #{subclass-axiom}}
       :deletions {:subclass-axioms #{subclass-axiom}}})))

(defn generalisation-compartment [subclass-axiom acc ontology]
  (let [id (entity-id subclass-axiom)
        sub-id (entity-id (as-class (.getSubClass subclass-axiom)))]
    (if (entity-has-rendition-by-id sub-id acc)
      {"Elements"
       {"Class"
        {sub-id {:compartments
                 (expressions-as-3-deep-text ontology #{(.getSuperClass subclass-axiom)} "SuperClasses")}}}
       :exported-axioms #{subclass-axiom}})))

(defn- get-annotation-stereotype [annotation]
  (let [annotation-property (.getProperty annotation)
        property-name-rendering (render annotation-property)]
    (cond
     (.isLabel annotation-property) "Label"
     (.isComment annotation-property) "Comment"
     :else property-name-rendering)))


(defn- get-annotation-value-rendition [annotation]
  (if (isa? (type (.getValue annotation)) OWLLiteral)
    (.. annotation (getValue) (getLiteral));pazaudē tipu
    (render (.getValue annotation))))

(defn- annotations-to-key-value-pairs [annotation-assertion]
  (map #(vector (str (.getIRI (.getProperty %)))
                (get-annotation-value-rendition %))
      (.getAnnotations annotation-assertion)))

(defn annotation-assertion-to-map [annotation-assertion]
  (into {:property (str (.getIRI (.getProperty annotation-assertion)))
         :value (get-annotation-value-rendition annotation-assertion)}
    (annotations-to-key-value-pairs annotation-assertion)))

(defn get-annotation-custom-render-spec [annotation object ontology]
  (if-let [ontology @lv.lumii.tda/*custom-render-spec-ontology-atom*]
		(let [annot-prop (.getProperty annotation)
	        annot-value (get-annotation-value-rendition annotation)
	        annotation-assertions (map annotation-assertion-to-map
	                                   (.getAnnotationAssertionAxioms annot-prop ontology))
	        owlgred-prefix "http://owlgred.lumii.lv/__plugins/fields/2011/1.0/owlgred#"
	        annot-prop-name (condp instance? object
	                          OWLClass          "ForClass_ShowAnnotation_InField"
	                          OWLIndividual     "ForIndiv_ShowAnnotation_InField"
	                          OWLObjectProperty "ForRole_ShowAnnotation_InField"
	                          OWLDataProperty   "ForAttr_ShowAnnotation_InField")
	        ;; need to filter by const value on specefic annotation
	        ;; eithor return list and filter later
	        ;; or pass annotation and filter here
	        ;; no only one annotation-assertion for each object,annot-prop,top-path possible
	        ;; so no way to add different conditions, if needed then different structure
	        spec (first (filter #(and (= (:property %)
	                                     (str owlgred-prefix annot-prop-name))
	                                  (or (nil? (get % (str owlgred-prefix "ShowAnnotation_RequireValue")))
	                                      (= (get % (str owlgred-prefix "ShowAnnotation_RequireValue"))
	                                         annot-value)))
	                            annotation-assertions))]
	    (if spec
	      (into {} (for [[k v] {:top-field-path           :value
	                            :value-subfield-path      (str owlgred-prefix "ShowAnnotationValue_inField")
	                            :lang-subfield-path       (str owlgred-prefix "ShowAnnotationLanguage_inField")
	                            :value-equality-condition (str owlgred-prefix "ShowAnnotation_RequireValue")
	                            :const-value              (str owlgred-prefix "ShowAnnotation_CreateValue")}]
	                 [k (spec v)]))))))

(defn- assoc-with-wrap [m k v]
  (let [wrapped-val (if (and (string? k)
                             (re-matches #"ASFictitious.*" k)
                             (not (set? v)))
              #{v}
              v)]
    (assoc m k wrapped-val)))

(defn- assoc-in-with-wrap [m [k & ks] v]
  (if ks
    (let [current-val (get m k)
          new-val (if (set? current-val)
                    (set (map #(assoc-in-with-wrap % ks v) current-val))
                    (assoc-in-with-wrap current-val ks v))]
       (assoc-with-wrap m k new-val))
    (assoc-with-wrap m k v)))

;; vēl jāpieliek cheks vai konfigurācija ir pareiza, ir visi sagaidāmie ceļi
;; ja nav tad jāliek paraistais
;; vēl arī jāpārtaisa parastais lai neliktu tik dziļi
(defn custom-annotation-rendition [annotation object ontology]
  (let [{:keys [top-field-path          
                value-subfield-path     
                lang-subfield-path      
                value-equality-condition
                const-value]
         :as spec}
        (get-annotation-custom-render-spec annotation object ontology)
        top-path-steps (split top-field-path #"/")
        value (or const-value
                  (get-annotation-value-rendition annotation))
        language (let [value (.getValue annotation)]
                     (if (and (isa? (type value) OWLLiteral)
                              (.hasLang value))
                       (.getLang value)))
        tmp (atom {})]
    (if lang-subfield-path
      (swap! tmp assoc-in-with-wrap (concat top-path-steps (split lang-subfield-path #"/")) language))
    (if value-subfield-path
      (swap! tmp assoc-in-with-wrap (concat top-path-steps (split value-subfield-path #"/")) value)
      (swap! tmp assoc-in-with-wrap top-path-steps value))))

(defn- annotation-as-map [annotation]
  {"AnnotationType" (get-annotation-stereotype annotation)
   "ValueLanguage" {"Value" (get-annotation-value-rendition annotation)
                    "Language" (let [value (.getValue annotation)]
                                 (if (and (isa? (type value) OWLLiteral)
                                          (.hasLang value))
                                   (.getLang value)))}})

(defn annotation-rendition [annotation object ontology]
  (let [spec (get-annotation-custom-render-spec annotation object ontology)]
    (if spec
      (custom-annotation-rendition annotation object ontology)
      {"ASFictitiousAnnotation" #{{"Annotation" (annotation-as-map annotation)}}})))

(defn- annotation-axiom-as-box-helper [annotation-axiom]
  (let [annotation (.getAnnotation annotation-axiom)
        id (entity-id annotation)]
    {"Elements"
     {"Annotation"
      {id {:compartments
           (annotation-as-map annotation)}}}}))

(defn annotation-property-box [annotation-property ontology]
  (let [id (entity-id annotation-property)]
    [{"Elements"
         {"AnnotationProperty"
          {id {:compartments
               {"Name" (name-with-namespace ontology annotation-property "Name" "Namespace")}}}}}
     (declaration-axioms annotation-property ontology)]))

(defn annotation-as-box [annotation _ _]
  [(let [id (entity-id annotation)]
     {"Elements"
      {"Annotation"
       {id {:compartments
            (annotation-as-map annotation)}}}})
   {:exported-axioms #{annotation}}])

(defn annotation-axiom-as-compartment [annotation-axiom]
  (let [annotation (.getAnnotation annotation-axiom)]
    {"Annotation" #{(annotation-as-map annotation)}}))

(defn property-annotation-axiom-as-compartment [id property annotation-axiom _ prefix expression-name path-builder-fn ontology]
  (let [annotation (.getAnnotation annotation-axiom)]
    [(assoc-in {}
               (path-builder-fn id)
               (annotation-rendition annotation property ontology))
     {:exported-axioms #{annotation-axiom}}]))

(defn data-property-annotation-axiom-as-compartment [id property annotation-axiom _ prefix expression-name path-builder-fn ontology]
  (let [annotation (.getAnnotation annotation-axiom)]
    [(assoc-in {}
               (path-builder-fn id)
               (annotation-rendition annotation property ontology))
     {:exported-axioms #{annotation-axiom}}]))

(defn property-chain-axiom-as-compartment [id property property-chain-axiom _ prefix expression-name path-builder-fn]
  (let [property-expression-list (.getPropertyChain property-chain-axiom)]
		[(assoc-in {} (path-builder-fn id)
		  	{"PropertyChains"
		 			{"ASFictitiousPropertyChains"
						#{{"PropertyChains"
					 	  	{"ASFictitiousPropertyChain"
						 			(seq (map (fn [property-chain-expression]
															(if (instance? org.semanticweb.owlapi.model.OWLObjectInverseOf property-chain-expression)
																{"PropertyChain" {"Property" (render (.getInverse property-chain-expression))
																									"Inverse" true}}
																{"PropertyChain" {"Property" (render property-chain-expression)}}))
													property-expression-list))}}}}})
     {:exported-axioms #{property-chain-axiom}}]))

(defn- get-annotation-declaration-axioms [annotation subject ontology]
  (set (filter #(= annotation (.getAnnotation %)) (.getAnnotationAssertionAxioms ontology (.getIRI subject)))))

(defn object-annotation-as-compartment [[individual annotation] _ ontology]
  [{:exported-axioms (get-annotation-declaration-axioms annotation individual ontology)}
   (individual-box individual ontology)
   (assoc-in {} ["Elements" "Object" (entity-id individual) :compartments] (annotation-rendition annotation individual ontology))])

(defn class-annotation-as-compartment [[class annotation] _ ontology]
  [{:exported-axioms (get-annotation-declaration-axioms annotation class ontology)}
   (class-box class ontology)
   (assoc-in {} ["Elements" "Class" (entity-id class) :compartments] (annotation-rendition annotation class ontology))])

(defn- min-cardinality [some? min]
  (cond
   (nil? min) (if some? 1)
   (coll? min) (apply max (if some? 1 0) min)
   :else (max (if some? 1 0) min)))

(defn- max-cardinality [some? max]
  (cond
   (nil? max) (if some? "*")
   (coll? max) (apply min max)
   :else max))

(defn- exact-cardinality [exact]
  (cond
   (nil? exact) nil
   (coll? exact) (apply min exact)
   :else exact))

(defn- cardinality-as-compartments [some min max exact]
  (let [min-value (min-cardinality some min)
        max-value (max-cardinality some max)
        exact (exact-cardinality exact)]
    (if exact
      {"min" exact "max" exact}
      (cond
       (and (nil? min-value) (not (nil? max-value))) {"min" (if some 1 0) "max" max-value}
       (and (not (nil? min-value)) (nil? max-value)) {"min" min-value "max" "*"}
       (and (not (nil? min-value)) (not (nil? max-value))) {"min" min-value "max" max-value}))))

(defn- cardinality-as-text-from-compartments [compartments]
  (if compartments
    (let [min (get compartments "min" "0")
          max (get compartments "max" "*")]
      (if (= min max)
        (str min)
        (str min ".." max)))))

(defn multiplicity-as-text [{:keys [min max exact]}]
  (let [exact (exact-cardinality exact)
	min (or exact (min-cardinality false min) 0)
	max (or exact (max-cardinality false max) "*")]
    (if (= min max)
      (str min)
      (str min ".." max))))

(defn object-property-restrinction-as-line [[class-id filler-id property
                                             {:keys [some axioms all min max exact] :or {some false all false}} :as line_description] _ ontology]
  [(assoc-in {} ["Elements" "Restriction" (entity-id line_description)]
             {:source class-id
              :target filler-id
              :compartments {"Name" (name-with-namespace ontology property "Role" "Namespace")
                             "Only" all
                             "Some" some
                             "Multiplicity" (cardinality-as-text-from-compartments (cardinality-as-compartments some min max exact))}})
   {:exported-axioms axioms}])

(defn data-type-box [data-type _ ontology]
  (let [definition-axioms (set (.getDatatypeDefinitions ontology data-type))
        ranges (set (map (fn [axiom] (.getDataRange axiom)) definition-axioms))
        first-range (first ranges)
        rest-ranges (rest ranges)
        id (entity-id data-type)
        data-type-name-structure (name-with-namespace ontology data-type "Name" "Namespace")
        create-data-type (fn [id range]
                            {"Elements"
                              {"DataType"
                               {id
                                {:compartments
                                 {"Label" "<<DataType>>"
                                  "Name" data-type-name-structure
                                  "DataTypeDefinition" (if range 
                                                          (render-with-namespace-in-curly ontology range))}}}}})]
    
    [(create-data-type id first-range)
      (if-not (empty? rest-ranges)
          (set (map (fn [range]
                       (create-data-type (entity-id range) range))
                    rest-ranges)))
     (declaration-axioms data-type ontology)
     {:rendered-entities #{id}}
     {:exported-axioms definition-axioms}]))

(defn has-key-axiom-as-compartment [has-key-axiom _ ontology]
  (let [class (as-class (.getClassExpression has-key-axiom))
        property-experssion (first (.getPropertyExpressions has-key-axiom))]
    [(class-box class ontology)
     {"Elements"
      {"Class"
       {(entity-id class) 
        {:compartments
         {"ASFictitiousKeys"
					 #{{"Keys" 
							{"ASFictitiousKey"
			          #{{"Key" 
			             (if (.isAnonymous property-experssion)
			               {"Property" (render-with-namespace-in-curly ontology (first (.getObjectPropertiesInSignature property-experssion)))
			                "Inverse" true}
			               {"Property" (render-with-namespace-in-curly ontology property-experssion)})}}}}}}}}}}
     {:exported-axioms #{has-key-axiom}}]))