(ns
    #^{:author "renarl",
       :doc "utilities for export"}
  lv.lumii.tda.export.utilities
  (:use lv.lumii.tda)
	(:require [clojure.string :as s])
  (:import [org.semanticweb.owlapi.model OWLEntity OWLClass IRI
            OWLObjectMinCardinality OWLObjectMaxCardinality]
           [org.jgrapht.graph SimpleGraph DefaultEdge]
           [org.jgrapht.alg BronKerboschCliqueFinder]
		   		 [uk.ac.manchester.cs.owl.owlapi.mansyntaxrenderer ManchesterOWLSyntaxOWLObjectRendererImpl]
		   		 [org.semanticweb.owlapi.util ShortFormProvider]
					 javax.swing.JOptionPane
					 javax.swing.SwingUtilities
					 java.awt.Frame))

(defn get-owl-thing []
  (.. OWLOntologyManager (getOWLDataFactory) (getOWLThing)))

(defn get-owl-top-obj-prop []
  (.. OWLOntologyManager (getOWLDataFactory) (getOWLTopObjectProperty)))

(defn get-owl-top-data-prop []
  (.. OWLOntologyManager (getOWLDataFactory) (getOWLTopDataProperty)))

(defn get-annotation-property [iri-str]
  (.. OWLOntologyManager (getOWLDataFactory) (getOWLAnnotationProperty (IRI/create iri-str))))

(defn get-annotation-assertion-axioms [something ontology annotation-property-iri-str]
  (filter #(= (.getProperty %)
              (IRI/create annotation-property-iri-str))
    (.getAnnotationAssertionAxioms something ontology)))

; testam viss pārsaukts uz kaut ko vienkāršu
#_(defn render
  "returns textual rendering of an owl object"
  [owl-object]
  (.getRendering OWLModelManager owl-object))
(defn render [entity]
	(let [renderer (new org.semanticweb.owlapi.util.SimpleRenderer)
				ontology (var-get (find-var 'lv.lumii.tda/*ontology*))]
;		(.setPrefixesFromOntologyFormat renderer ontology OWLOntologyManager true)
;		(.setShortFormProvider renderer (new ManchesterOWLSyntaxPrefixNameShortFormProvider OWLOntologyManager ontology))
		(.setShortFormProvider renderer (new org.semanticweb.owlapi.util.SimpleShortFormProvider))
		(let [rendering (.render renderer entity)]
			(or (last (re-matches #"^<?([^>]*)>?.*" rendering)) ""))))

; problēmas ar s/tail pagaidām aizstājam ar tukšo
#_(defn namespace-rendering-without-owl-extension [namespace-str]
  (if (= (s/tail namespace-str 4) ".owl")
    (s/butlast namespace-str 4)
    namespace-str))
(defn namespace-rendering-without-owl-extension [namespace-str]
  namespace-str)

(defn get-namespace [entity-uri-str]
	(let [entity-uri (new java.net.URI entity-uri-str)
		  fragment (.getFragment entity-uri)
		  path (.getPath entity-uri)]
		(if fragment 
			(str (.getHost entity-uri) path)
			(if path 
				(str (.getHost entity-uri) (-> path
										 	   (s/split #"/")
										 	   (butlast)
										 	   (->> (interpose "/"))
										 	   (->> (apply str)))
										 	  "/")
				entity-uri-str))))

(def default-prefix-list
	["http://www.w3.org/2006/12/owl2-xml#"
	 "http://www.w3.org/2002/07/owl#"
	 "http://www.w3.org/1999/02/22-rdf-syntax-ns#"
	 "http://www.w3.org/2000/01/rdf-schema#"
	 "http://www.w3.org/2001/XMLSchema#"
	 "http://www.w3.org/XML/1998/namespace"
	 "http://www.w3.org/2001/XMLSchema#"])

(defn drop-last-slash [s]
	(if (= "/" (str (last s)))
		(apply str (butlast s))
		s))

(defn prefix-name-to-prefix-map [ontology]
	(map (fn [[prefix iri]]
	 			[(apply str (butlast prefix)) iri])
			(.. OWLOntologyManager (getOntologyFormat ontology) (asPrefixOWLOntologyFormat) (getPrefixName2PrefixMap))))

(defn short-prefix-name [ontology iri]
	(let [name-space (.getStart iri)]
		(some (fn [[k v]] (if (= v name-space) k)) (prefix-name-to-prefix-map ontology))))

;
(defn- has-default-prefix? [iri]
	(let [prefix (.getStart iri)]
		(some #(.startsWith prefix %) default-prefix-list)))

(def owlgred-recognized-names-from-default-namespaces
	[;data-property-types
	 "Literal"
 	 "NCName" 
	 "NMTOKEN" 
	 "Name" 
	 "PlainLiteral"
	 "XMLLiteral" 
	 "anyURI" 
	 "base64Binary" 
	 "boolean" 
	 "byte"
	 "dateTime"
	 "dateTimeStamp"
     "decimal" 
	 "double" 
	 "float"
	 "hexBinary" 
	 "int" 
	 "integer"
	 "language" 
	 "long" 
	 "negativeInteger"
	 "nonNegativeInteger"
	 "nonPositiveInteger" 
	 "normalizedString"
	 "positiveInteger" 
	 "rational" 
	 "real" 
	 "short" 
	 "string" 
	 "token"
	 "unsignedByte"
	 "unsignedInt" 
	 "unsignedLong" 
	 "unsignedShort"
	 ; annotation properties
	 "backwardCompatibleWith"
	 "comment"
	 "deprecated"
	 "incompatibleWith" 
	 "isDefinedBy" 
	 "label"
	 "priorVersion" 
	 "seeAlso" 
	 "versionInfo"
	 ; classes
	 "Thing"
	 "Nothing"])

(defn- has-recognized-name? [iri]
	(let [fragment (.getFragment iri)]
		(some #(= fragment %) owlgred-recognized-names-from-default-namespaces)))

(defn has-short-rendering-in-owlgred? [iri]
	(and (has-default-prefix? iri)
		 (has-recognized-name? iri)))

(defn simple-short-form-from-iri [iri]
	(let [uri (.toURI iri)]
		(if-let [fragment (.getFragment uri)]
			fragment
			(if-let [path (.getPath uri)]
				(.substring path (+ (.lastIndexOf path "/") 1))
				(.toString uri)))))

(defn short-namespace-name [ontology]
	(let [ontology-iri (.getOntologyIRI (.getOntologyID ontology))]
     (if-let [iri ontology-iri]
     	(let [ad-hoc-short-form (simple-short-form-from-iri ontology-iri)]
	        (if-let [short-form (short-prefix-name ontology iri)]
						(if (not (= short-form ""))
							short-form
							ad-hoc-short-form)
						ad-hoc-short-form)
					ad-hoc-short-form))))


(defn get-entity-rendering-name-namespace [ontology entity]
  (if (instance? org.semanticweb.owlapi.model.OWLObjectInverseOf entity)
	{:rendering (render entity)
	 :name (render entity)
	 :namespace ""
	 :iri ""}
	(let [entity-uri (.toURI (.getIRI entity))
        namespace (get-namespace (str entity-uri))
        ontology-namespace (let [ontology-iri (.getOntologyIRI (.getOntologyID ontology))]
                             (if ontology-iri
                               (let [ontology-uri (.toURI ontology-iri)]
 																	(str (.getHost ontology-uri) (.getPath ontology-uri)))
                               ""))

        entity-name (render entity)]
    (if (or (= namespace ontology-namespace) (has-recognized-name? (.getIRI entity)))
			{:rendering (render entity)
			 :name entity-name
			 :namespace ""
			 :iri (.getIRI entity)}
			{:rendering nil
			 :name entity-name
			 :namespace (or (short-prefix-name ontology (.getIRI entity))
											(namespace-rendering-without-owl-extension (let [rendering (render (IRI/create (drop-last-slash namespace)))]
																															 (if-not (= rendering "<Error! null>")
																																	rendering
																																	""))))
			 :iri (.getIRI entity)}))))


(defn render-with-namespace-in-curly [ontology owl-object]
	(let [renderer (doto (new ManchesterOWLSyntaxOWLObjectRendererImpl)
										(.setShortFormProvider (proxy [ShortFormProvider] [] 
												(getShortForm [owl-entity]
													 (let [{:keys [rendering name namespace iri]} (get-entity-rendering-name-namespace ontology owl-entity)]
															(if (or rendering (has-short-rendering-in-owlgred? iri))
																name
																(str name
																		"{" namespace "}")))))))]
		(.render renderer owl-object)))


(let [i (atom 0)
      t (atom {})]
  (defn- generate-unique-id
    "Returns a distinct numeric ID for each call."
    []
    (swap! i inc))
	(defn get-entity-by-id [id]
    "returns entity with the given id"
    []
    (some (fn [[entity entity-id]] (if (= entity-id id) entity)) @t))
  (defn reset-id-map
    "Reset entity id map"
    []
		(swap! i (fn [_] 0))
    (swap! t (fn [_] {})))
  (defn entity-id [entity]
    "returns unique string ID of each entity"
    (let [current-id (get (deref t) entity)]
      (if current-id
        current-id
        (let [new-id (generate-unique-id)
              new-id-str (str new-id)]
          (do (assert new-id)
              (swap! t assoc-in [entity] new-id-str)
              new-id-str))))))

(defn add-ontology-imports-closure! [m ontology]
  (let [ontology-id (entity-id ontology)]
    (if-not (get @m ontology-id)
      (do (swap! m #(assoc % ontology-id ontology))
          (doseq [o (.getImports ontology)]
            (add-ontology-imports-closure! m o)))))
  @m)

(defn- get-property-end-helper
  ([property ontology end-getter-fn] (get-property-end-helper property ontology end-getter-fn (atom #{ontology})))
  ([property ontology end-getter-fn visited-ontologies]
    (let [ends (end-getter-fn property ontology)]
      (if-not (empty? ends)
        ends
        (let [imports (.getImports ontology)
              ends-from-import (some (fn [end]
                                         (if-not (empty? end) end))
                                       (map #(if-not (some #{%} @visited-ontologies)
                                                (do (swap! visited-ontologies (fn [visited] (conj visited %)))
                                                  (get-property-end-helper property % end-getter-fn visited-ontologies))
                                                #{})
                                            imports))]
          (if ends-from-import
            ends-from-import
            #{}))))))

(defn get-property-domain [property ontology]
  (get-property-end-helper property ontology (fn [p o] (.getDomains p o))))

(defn get-property-range [property ontology]
  (get-property-end-helper property ontology (fn [p o] (.getRanges p o))))

(defn get-domain-of-property-pair [[property inverse-property] ontology]
	(let [ontology-imports-closure (.getImportsClosure ontology)
				domain (.getDomains property ontology-imports-closure)]
		(if inverse-property
			(.addAll domain (.getRanges inverse-property ontology-imports-closure)));inplace
		domain))

(defn get-range-of-property-pair [[property inverse-property] ontology]
	(let [ontology-imports-closure (.getImportsClosure ontology)
				range (.getRanges property ontology-imports-closure)]
		(if inverse-property
			(.addAll range (.getDomains inverse-property ontology-imports-closure)));inplace
		range))

(defn declaration-axioms
  "get declaration axioms for entity"
  [entity ontology]
  {:exported-axioms (set (.getDeclarationAxioms ontology entity))})

(defn pairs-of-inverse-object-properties
  "set of property pairs, where in each pair properties are inverse to each other and have invrese domains and ranges"
  [ontology]
  (let [ontology-imports-closure (.getImportsClosure ontology)
				properties (remove #(= % (get-owl-top-obj-prop)) (.getObjectPropertiesInSignature ontology))
				property-in-pair (atom #{})]
    (set (map seq
              (reduce
               (fn [acc property]
	               (if-not (contains? property-in-pair property)
										(let [inverse-properties (.getInverses property ontology-imports-closure)
												 inverse-property (first inverse-properties)
												 inv-inv-properties (if inverse-property 
																								(.getInverses inverse-property ontology-imports-closure)
																								[])]
	                   (if (and (= (count inverse-properties) 1)
															(= (count inv-inv-properties) 1)
															(= property (first inv-inv-properties)))
	                     (let [range (.getRanges property ontology-imports-closure)
														 domain (.getDomains property ontology-imports-closure)
													 
	                           inv-range (.getRanges  inverse-property ontology-imports-closure)
	                           inv-domain (.getDomains inverse-property ontology-imports-closure)]
	                       (if (and
	                            (or (empty? range) (empty? inv-domain) (= range inv-domain))
	                            (or (empty? domain) (empty? inv-range) (= domain inv-range)))
	                         (if-not (some #(or (= % [property inverse-property])
																							(= % [inverse-property property]))
																				 acc)
															(do (swap! property-in-pair conj property inverse-property)
																	(conj acc [property inverse-property]))
															acc)
	                         acc))
	                     acc))
										acc))
               #{}
               properties)))))

(defn object-property-redered-as-inverse? [acc property]
  (some (fn [[_ inv-property]] (= property inv-property)) (:object-properties acc)))

(defn- subclass-relation-rendered-as-generalization-through-fork? [acc subclass-id superclass-id]
    (some (fn [[_ {{super-id "GeneralizationToFork"} :outgoing
                   {sub-ids "AssocToFork"} :incomming}]]
              (and (= super-id superclass-id)
                   (some #{subclass-id} sub-ids)))
      (get-in acc ["Elements" "HorizontalFork"])))

(defn- subclass-relation-rendered-as-generalization? [acc subclass-id superclass-id]
    (some (fn [[_ {:keys [source target]}]]
              (and (= source subclass-id)
                   (= target superclass-id)))
      (get-in acc ["Elements" "Generalization"])))

(defn subclass-relation-rendered-graphicaly? [acc subclass-id superclass-id]
  (or (subclass-relation-rendered-as-generalization? acc subclass-id superclass-id)
      (subclass-relation-rendered-as-generalization-through-fork? acc subclass-id superclass-id)))
  
(defn as-class [anything]
  (cond
    (instance? java.util.TreeSet anything) (as-class (set anything))
    (not (coll? anything)) (as-class #{anything})
    (empty? anything) (as-class (get-owl-thing))
    (and (= (count anything) 1) (instance? OWLClass (first anything))) (first anything)
    :else anything))

(defn axioms-with-type [axioms axiom-class]
  (set (filter
        (fn [axiom]
          (instance? axiom-class axiom))
        axioms)))

(defn ontology-axioms-with-type [ontology axiom-class]
  (axioms-with-type (.getAxioms ontology) axiom-class))

(defn get-all-pairs [coll]
  (if (> (count coll) 1)
    (let [head (first coll)]
      (concat (map (fn [a] [head a]) (rest coll))
              (get-all-pairs (rest coll))))))

(defn entity-has-rendition-by-id [entity-id acc]
  (some #{entity-id} (:rendered-entities acc)))

(defn axiom-keyword-for-group [group-name]
  (keyword (str (name group-name) "-axioms")))

(defn groups-for-axiom-type [axiom-type entity-getter group-name]
  (fn [_ ontology]
    (let [axioms (ontology-axioms-with-type ontology axiom-type)]
      (set (map (fn [axiom]
             (let [entities (entity-getter axiom)
                   pairs (get-all-pairs entities)]
               (map (fn [[entity-1 entity-2]]
                      (let [p #{entity-1 entity-2}]
                        {group-name #{p}
                         (axiom-keyword-for-group group-name) {p #{axiom}}}))
                    pairs)))
           axioms)))))

(defn clique-getter-for-group-name [group-name]
   (fn [acc _]
     (let [graph (new SimpleGraph DefaultEdge)
           entity-pairs (map seq (group-name acc))]
       (doseq [[entity-1 entity-2] entity-pairs]
         (do (.addVertex graph entity-1)
             (.addVertex graph entity-2)
             (.addEdge graph entity-1 entity-2)))
       (map set (.getAllMaximalCliques (new BronKerboschCliqueFinder graph))))))

(defn group-joiner-by-clique-for-group-name [group-name]
  (fn [clique acc _]
    (let [pairs (set (map set (get-all-pairs clique)))]
      {:deletions #{{group-name pairs}
                    (set (map (fn [p]
                                {(axiom-keyword-for-group group-name)
                                 {p (get-in acc [(axiom-keyword-for-group group-name) p])}})
                              pairs))}
       :additions #{{group-name #{clique}}
                    (set (map (fn [p]
                                {(axiom-keyword-for-group group-name)
                                 {clique (get-in acc [(axiom-keyword-for-group group-name) p])}})
                              pairs))}})))

(defn id [e] e)

(defn dissoc-all-except [map key-set]
  (apply dissoc map (seq (apply disj (set (keys map)) (seq key-set)))))

(defn show-error-message [title message]
	#_(JOptionPane/showMessageDialog (SwingUtilities/getAncestorOfClass Frame (.getWorkspace OWLEditorKit))
																 message
																 title
																 JOptionPane/ERROR_MESSAGE))